package fr.isae.geometry;

import fr.isae.observer.Observable;

/**
 * <code>Point</code> definit une classe point mathematique dans un
 * plan qui peut est dans un repere cartesien.<BR>
 * Un point peut etre translate. Sa distance par rapport a un autre
 * point peut etre obtenue.
 *
 * @author <a href="mailto:garion@isae.fr">Christophe Garion</a>
 * @version 1.0
 */
public class Point extends Observable {
    private double x;
    private double y;

    /**
     * La valeur servant a comparer des <code>double</code> entre eux.
     */
    public static final double EPS = 10E-9;

    /**
     * Une constante representant le point a l'origine.
     */
    public static final Point ORIGINE = new Point(0.0, 0.0);

    /**
     * Cree une nouvelle instance de <code>Point</code>.
     *
     * @param x un <code>double</code> representant l'abscisse du
     *          point a creer
     * @param y un <code>double</code> representant l'ordonnee du
     *          point a creer
     */
    public Point(double x, double y) {
        super();
        this.x = x;
        this.y = y;
    }

    /**
     * La valeur de l'abscisse du point.
     *
     * @return un <code>double</code>  qui est l'abscisse du point
     */
    public double getX() {
        return this.x;
    }

    /**
     * Changer la valeur de l'abscisse du point.
     *
     * @param x un <code>double</code> qui est la nouvelle abscisse
     */
    public void setX(double x) {
        this.x = x;
        this.avertir();
    }

    /**
     * La valeur de l'ordonnee du point.
     *
     * @return un <code>double</code> qui est l'ordonnee du point
     */
    public double getY() {
        return this.y;
    }

    /**
     * Changer la valeur de l'ordonnee du point.
     *
     * @param y un <code>double</code> qui est la nouvelle ordonnee
     */
    public void setY(double y) {
        this.y = y;
        this.avertir();
    }

    /**
     * La valeur du module du point.
     *
     * @return un <code>double</code> qui est la valeur du module.
     */
    public double getModule() {
        return (Math.sqrt(this.x * this.x + this.y * this.y));
    }

    /**
     * Changer la valeur du module du point.
     *
     * @param module la nouvelle valeur du module du point. <b>Elle
     *               doit etre positive.</b>
     */
    public void setModule(double module) {
        double ancienModule = getModule();

        if (ancienModule < EPS) {
            this.x = module;
        } else {
            this.x = (this.x * module) / ancienModule;
            this.y = (this.y * module) /  ancienModule;
        }
    }

    /**
     * La valeur de l'argument du point.
     *
     * @return un <code>double</code> qui est la valeur de l'argument.
     */
    public double getArgument() {
        if (this.getModule() < EPS) {
            return 0;
        }

        double ancienArg = Math.acos(this.x / this.getModule());

        if (this.y < 0) {
            return (-1.0 * ancienArg);
        }

        return ancienArg;
    }

    /**
     * Changer la valeur de l'argument du point.
     *
     * @param argument la nouvelle valeur de l'argument du point.
     */
    public void setArgument(double argument) {
        double ancienModule = this.getModule();
        x = ancienModule * Math.cos(argument);
        y = ancienModule * Math.sin(argument);
    }

    /**
     * <code>translater</code> permet de translater le point.
     *
     * @param dx un <code>double</code> qui represente l'abscisse du
     *           vecteur de translation
     * @param dy un <code>double</code> qui represente l'ordonnee du
     *           vecteur de translation
     */
    public void translater(double dx, double dy) {
        this.x = this.x + dx;
        this.y = this.y + dy;
        this.avertir();
    }

    /**
     * <code>afficher</code> permet d'afficher les coordonnees du point.
     */
    public void afficher() {
        System.out.println(this);
    }

    /**
     * <code>toString</code> renvoie un objet de type <code>String</code>
     * qui represente une chaine de caracteres representant le point.
     *
     * <p> Nous verrons plus tard (cf. cours sur l'heritage) que
     * cette methode nous permet d'utiliser un objet comme un type
     * primitif et d'ecrire par exemple <code>System.out.println("Le
     * point est : " + p)</code> ou <code>p</code> est une poignee
     * sur un objet de type <code>Point</code>.
     *
     * @return un objet de type <code>String</code> representant
     *         le point. Pour un point de coordonnees (2,3), cet objet
     *         representera la chaine <code>(2,3)</code>.
     */
    public String toString() {
        return ("(" + this.x + "," + this.y + ")");
    }

    /**
     * <code>distance</code> permet de calculer la distance entre deux
     * points.
     *
     * @param p un <code>Point</code> qui est l'autre point pour calculer
     *          la distance
     * @return un <code>double</code> qui est la distance entre les deux
     *         point
     */
    public double distance(Point p) {
        return (Math.sqrt(((this.x - p.x) * (this.x - p.x)) +
                          ((this.y - p.y) * (this.y - p.y))));
    }

    /**
     * <code>equals</code> permet de verifier si deux points sont egaux, i.e.
     * s'ils ont les memes coordonnees.
     *
     * @param o un <code>Object</code> qui est l'objet dont on veut verifier s'il
     *          est egal a <code>this</code>
     * @return un <code>boolean</code> qui est <code>true</code> si <code>o</code>
     *         est un point egal a <code>this</code>
     */
    @Override public boolean equals(Object o) {
        if (!(o instanceof Point)) {
            return false;
        }

        return (this.x == ((Point) o).x) && (this.y == ((Point) o).y);
    }

    /**
     * <code>clone</code> permet d'obtenir une copie de <code>this</code>. On obtient
     * un nouveau point avec des coordonnees identiques.
     *
     * @return un <code>Point p</code> tel que <code>this != p</code> et
     *         <code>this.equals(p)</code>
     */
    public Point clone() {
        return new Point(this.x, this.y);
    }
}
